* Replication files for
* "Capital Reallocation and the Cyclicality of Aggregate Productivity" by Russ Cooper & Immo Schott
* Prepared: May 2023

use if year >= 1971 using "$datapath/microdata_coopimmo.dta", clear

***************************************************************
***************************************************************
* Create macro-level data set
***************************************************************
***************************************************************

* Collect moments
local means "(mean)"
local sums "(sum)"

* Collect sums
local vars aqc sppe capx sale ppent R emp
foreach x of local vars {
	local sums "`sums' `x'_sum = `x'"	
}

* Collect means
local vars sd_TFP sd_APK adj_R alphahat alpha beta TFP
foreach x of local vars {
	local means "`means' `x'_mean = `x'"	
}

* Collapse moments by year
collapse `means' `sums', by(year)

tsset year

* Clarify where necessary
ren R_sum R_micro 
lab var R_micro "Sum of micro-level reallocation"
ren adj_R R_frac
lab var R_frac "Fraction of firms with R > 0"
ren *_mean *

* Label key variables
lab var sd_TFP "Dispersion in plant-level TFP"
lab var sd_APK "Dispersion in plant-level APK"
lab var alphahat "Elasticity of effective capital"

order year R_frac sd_TFP R_micro

***************************************************************
***************************************************************
* Compute macro-level moments
***************************************************************
***************************************************************

* Redefine reallocation at the aggregate level 

gen R_macro = aqc_sum + sppe_sum
lab var R_macro "Reallocation, aggregate"
gen I = capx_sum + aqc_sum
lab var I "Investment, aggregate"

* Capital reallocation as a fraction of total capital or assets
gen RK = R_macro / ppent_sum
lab var RK "Reallocation over capital, aggregate"

* Capital reallocation as a fraction of investment
gen RI  = R_macro / I
lab var RI "Reallocation over investment, aggregate"

drop aqc_sum sppe_sum capx_sum I

* Aggregate productivity as Atilde, following the model
gen Atilde = sale_sum / (ppent_sum^alpha * emp_sum^beta)
drop sale_sum ppent_sum emp_sum alpha beta 
lab var Atilde "Aggregate productivity"

* Following CHW_2007, use the average of micro-level TFP
bys year: egen TFP_agg = mean(TFP)
drop TFP 
ren TFP TFP
lab var TFP "Aggregate TFP, from micro-level"

* Alternative measures from outside Compustat
* a) Aggregate real GDP (from FRED)
* b) Fernald's TFP series
merge 1:1 year using "$datapath/productivity_data.dta", keep(match) nogen
lab var dtfp "Change in TFP, Basu et al"
lab var dtfp_util "Change in TFP (utility adj.), Basu et al"
lab var GDPr "GDP, real"
lab var year "Time"

order year R_frac RK sd_TFP R_macro R_micro


* Create cyclical variables

* Reallocation R_macro is in USD and needs to be deflated
ren year fyear
merge 1:1 fyear using "$datapath/prices_updated.dta", keep(match) nogen keepusing(gdp_ind)
ren fyear year 
gen logR = log(R_macro/gdp_ind)

gen logAtilde 	= log(Atilde)
gen logGDP 	= log(GDPr)

* De-trend
local varlist logR R_frac RK sd_APK logAtilde logGDP
foreach x of local varlist{
	* Detrended version 
	qui reg `x' year 
	predict `x'_res, res
	* HP-filtered version
	tsfilter hp `x'_hp = `x', s(100)
}

* Create aggregate data set
compress
save "$datapath/macrodata_coopimmo.dta", replace
